<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TestCase;
use App\Models\Currency;
use App\Models\ExchangeRate;
use App\Models\User;

class ExchangeRateTest extends TestCase
{
    use RefreshDatabase;

    protected $fromCurrency;
    protected $toCurrency;
    protected $user;

    /**
     * الإعداد قبل كل اختبار
     */
    protected function setUp(): void
    {
        parent::setUp();
        
        // إنشاء عملات للاختبار
        $this->fromCurrency = Currency::create([
            'name' => 'الدولار الأمريكي',
            'code' => 'USD',
            'symbol' => '$',
            'country' => 'الولايات المتحدة',
            'is_active' => true,
            'decimal_places' => 2,
            'sort_order' => 1
        ]);

        $this->toCurrency = Currency::create([
            'name' => 'اليورو',
            'code' => 'EUR',
            'symbol' => '€',
            'country' => 'الاتحاد الأوروبي',
            'is_active' => true,
            'decimal_places' => 2,
            'sort_order' => 2
        ]);
        
        // إنشاء مستخدم للاختبار
        $this->user = User::factory()->create();
    }

    /**
     * اختبار عرض قائمة أسعار الصرف
     */
    public function test_index_displays_exchange_rates_list()
    {
        // إنشاء سعر صرف للاختبار
        ExchangeRate::create([
            'from_currency_id' => $this->fromCurrency->id,
            'to_currency_id' => $this->toCurrency->id,
            'buy_rate' => 0.85,
            'sell_rate' => 0.87,
            'is_active' => true,
            'effective_date' => now(),
            'updated_by' => $this->user->id
        ]);

        // زيارة صفحة قائمة أسعار الصرف
        $response = $this->get(route('exchange-rates.index'));

        // التحقق من نجاح الاستجابة
        $response->assertStatus(200);

        // التحقق من وجود سعر الصرف في الصفحة
        $response->assertSee('USD');
        $response->assertSee('EUR');
        $response->assertSee('0.85');
        $response->assertSee('0.87');
    }

    /**
     * اختبار إنشاء سعر صرف جديد
     */
    public function test_store_creates_new_exchange_rate()
    {
        // بيانات سعر الصرف الجديد
        $exchangeRateData = [
            'from_currency_id' => $this->fromCurrency->id,
            'to_currency_id' => $this->toCurrency->id,
            'buy_rate' => 0.90,
            'sell_rate' => 0.92,
            'is_active' => true,
            'effective_date' => now()->format('Y-m-d H:i:s'),
            'notes' => 'سعر صرف تجريبي'
        ];

        // إرسال طلب إنشاء سعر صرف جديد
        $response = $this->actingAs($this->user)
            ->post(route('exchange-rates.store'), $exchangeRateData);

        // التحقق من إعادة التوجيه بعد الإنشاء
        $response->assertRedirect(route('exchange-rates.index'));

        // التحقق من وجود سعر الصرف في قاعدة البيانات
        $this->assertDatabaseHas('exchange_rates', [
            'from_currency_id' => $this->fromCurrency->id,
            'to_currency_id' => $this->toCurrency->id,
            'buy_rate' => 0.90,
            'sell_rate' => 0.92,
            'updated_by' => $this->user->id
        ]);
    }

    /**
     * اختبار عرض تفاصيل سعر صرف
     */
    public function test_show_displays_exchange_rate_details()
    {
        // إنشاء سعر صرف للاختبار
        $exchangeRate = ExchangeRate::create([
            'from_currency_id' => $this->fromCurrency->id,
            'to_currency_id' => $this->toCurrency->id,
            'buy_rate' => 0.85,
            'sell_rate' => 0.87,
            'is_active' => true,
            'effective_date' => now(),
            'updated_by' => $this->user->id,
            'notes' => 'ملاحظات اختبار'
        ]);

        // زيارة صفحة تفاصيل سعر الصرف
        $response = $this->get(route('exchange-rates.show', $exchangeRate->id));

        // التحقق من نجاح الاستجابة
        $response->assertStatus(200);

        // التحقق من وجود تفاصيل سعر الصرف في الصفحة
        $response->assertSee('USD');
        $response->assertSee('EUR');
        $response->assertSee('0.85');
        $response->assertSee('0.87');
        $response->assertSee('ملاحظات اختبار');
    }

    /**
     * اختبار تحديث سعر صرف
     */
    public function test_update_modifies_exchange_rate()
    {
        // إنشاء سعر صرف للاختبار
        $exchangeRate = ExchangeRate::create([
            'from_currency_id' => $this->fromCurrency->id,
            'to_currency_id' => $this->toCurrency->id,
            'buy_rate' => 0.85,
            'sell_rate' => 0.87,
            'is_active' => true,
            'effective_date' => now(),
            'updated_by' => $this->user->id
        ]);

        // بيانات التحديث
        $updatedData = [
            'from_currency_id' => $this->fromCurrency->id,
            'to_currency_id' => $this->toCurrency->id,
            'buy_rate' => 0.88,
            'sell_rate' => 0.90,
            'is_active' => true,
            'effective_date' => now()->format('Y-m-d H:i:s'),
            'notes' => 'سعر صرف محدث'
        ];

        // إرسال طلب تحديث سعر الصرف
        $response = $this->actingAs($this->user)
            ->put(route('exchange-rates.update', $exchangeRate->id), $updatedData);

        // التحقق من إعادة التوجيه بعد التحديث
        $response->assertRedirect(route('exchange-rates.index'));

        // التحقق من تحديث سعر الصرف في قاعدة البيانات
        $this->assertDatabaseHas('exchange_rates', [
            'id' => $exchangeRate->id,
            'buy_rate' => 0.88,
            'sell_rate' => 0.90,
            'notes' => 'سعر صرف محدث'
        ]);
    }

    /**
     * اختبار حذف سعر صرف
     */
    public function test_destroy_deletes_exchange_rate()
    {
        // إنشاء سعر صرف للاختبار
        $exchangeRate = ExchangeRate::create([
            'from_currency_id' => $this->fromCurrency->id,
            'to_currency_id' => $this->toCurrency->id,
            'buy_rate' => 0.85,
            'sell_rate' => 0.87,
            'is_active' => true,
            'effective_date' => now(),
            'updated_by' => $this->user->id
        ]);

        // إرسال طلب حذف سعر الصرف
        $response = $this->delete(route('exchange-rates.destroy', $exchangeRate->id));

        // التحقق من إعادة التوجيه بعد الحذف
        $response->assertRedirect(route('exchange-rates.index'));

        // التحقق من حذف سعر الصرف من قاعدة البيانات
        $this->assertDatabaseMissing('exchange_rates', [
            'id' => $exchangeRate->id
        ]);
    }

    /**
     * اختبار تغيير حالة سعر الصرف
     */
    public function test_toggle_status_changes_exchange_rate_status()
    {
        // إنشاء سعر صرف للاختبار
        $exchangeRate = ExchangeRate::create([
            'from_currency_id' => $this->fromCurrency->id,
            'to_currency_id' => $this->toCurrency->id,
            'buy_rate' => 0.85,
            'sell_rate' => 0.87,
            'is_active' => true,
            'effective_date' => now(),
            'updated_by' => $this->user->id
        ]);

        // التحقق من الحالة الأولية
        $this->assertTrue($exchangeRate->is_active);

        // إرسال طلب تغيير الحالة
        $response = $this->actingAs($this->user)
            ->post(route('exchange-rates.toggle-status', $exchangeRate->id));

        // التحقق من إعادة التوجيه بعد تغيير الحالة
        $response->assertRedirect(route('exchange-rates.index'));

        // التحقق من تغيير الحالة في قاعدة البيانات
        $this->assertDatabaseHas('exchange_rates', [
            'id' => $exchangeRate->id,
            'is_active' => false
        ]);
    }

    /**
     * اختبار الحصول على سعر الصرف بين عملتين
     */
    public function test_get_rate_returns_exchange_rate()
    {
        // إنشاء سعر صرف للاختبار
        ExchangeRate::create([
            'from_currency_id' => $this->fromCurrency->id,
            'to_currency_id' => $this->toCurrency->id,
            'buy_rate' => 0.85,
            'sell_rate' => 0.87,
            'is_active' => true,
            'effective_date' => now(),
            'updated_by' => $this->user->id
        ]);

        // إرسال طلب للحصول على سعر الصرف
        $response = $this->post(route('exchange-rates.get-rate'), [
            'from_currency_id' => $this->fromCurrency->id,
            'to_currency_id' => $this->toCurrency->id
        ]);

        // التحقق من نجاح الاستجابة
        $response->assertStatus(200);

        // التحقق من البيانات المرجعة
        $responseData = $response->json();
        $this->assertEquals(0.85, $responseData['buy_rate']);
        $this->assertEquals(0.87, $responseData['sell_rate']);
        $this->assertEquals(false, $responseData['is_inverse']);
    }

    /**
     * اختبار تحديث أسعار الصرف بشكل جماعي
     */
    public function test_bulk_update_modifies_multiple_exchange_rates()
    {
        // إنشاء أسعار صرف للاختبار
        $exchangeRate1 = ExchangeRate::create([
            'from_currency_id' => $this->fromCurrency->id,
            'to_currency_id' => $this->toCurrency->id,
            'buy_rate' => 0.85,
            'sell_rate' => 0.87,
            'is_active' => true,
            'effective_date' => now(),
            'updated_by' => $this->user->id
        ]);

        // إنشاء عملة ثالثة
        $thirdCurrency = Currency::create([
            'name' => 'الجنيه الإسترليني',
            'code' => 'GBP',
            'symbol' => '£',
            'country' => 'المملكة المتحدة',
            'is_active' => true,
            'decimal_places' => 2,
            'sort_order' => 3
        ]);

        $exchangeRate2 = ExchangeRate::create([
            'from_currency_id' => $this->fromCurrency->id,
            'to_currency_id' => $thirdCurrency->id,
            'buy_rate' => 0.75,
            'sell_rate' => 0.77,
            'is_active' => true,
            'effective_date' => now(),
            'updated_by' => $this->user->id
        ]);

        // بيانات التحديث الجماعي
        $bulkUpdateData = [
            'rates' => [
                [
                    'id' => $exchangeRate1->id,
                    'buy_rate' => 0.88,
                    'sell_rate' => 0.90
                ],
                [
                    'id' => $exchangeRate2->id,
                    'buy_rate' => 0.78,
                    'sell_rate' => 0.80
                ]
            ]
        ];

        // إرسال طلب تحديث جماعي
        $response = $this->actingAs($this->user)
            ->post(route('exchange-rates.bulk-update'), $bulkUpdateData);

        // التحقق من نجاح الاستجابة
        $response->assertStatus(200);
        $response->assertJson(['success' => true]);

        // التحقق من تحديث أسعار الصرف في قاعدة البيانات
        $this->assertDatabaseHas('exchange_rates', [
            'id' => $exchangeRate1->id,
            'buy_rate' => 0.88,
            'sell_rate' => 0.90
        ]);

        $this->assertDatabaseHas('exchange_rates', [
            'id' => $exchangeRate2->id,
            'buy_rate' => 0.78,
            'sell_rate' => 0.80
        ]);
    }
}
