<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TestCase;
use App\Models\Currency;

class CurrencyTest extends TestCase
{
    use RefreshDatabase;

    /**
     * اختبار عرض قائمة العملات
     */
    public function test_index_displays_currencies_list()
    {
        // إنشاء عملات للاختبار
        Currency::create([
            'name' => 'الدولار الأمريكي',
            'code' => 'USD',
            'symbol' => '$',
            'country' => 'الولايات المتحدة',
            'is_active' => true,
            'decimal_places' => 2,
            'sort_order' => 1
        ]);

        Currency::create([
            'name' => 'اليورو',
            'code' => 'EUR',
            'symbol' => '€',
            'country' => 'الاتحاد الأوروبي',
            'is_active' => true,
            'decimal_places' => 2,
            'sort_order' => 2
        ]);

        // زيارة صفحة قائمة العملات
        $response = $this->get(route('currencies.index'));

        // التحقق من نجاح الاستجابة
        $response->assertStatus(200);

        // التحقق من وجود العملات في الصفحة
        $response->assertSee('الدولار الأمريكي');
        $response->assertSee('USD');
        $response->assertSee('اليورو');
        $response->assertSee('EUR');
    }

    /**
     * اختبار إنشاء عملة جديدة
     */
    public function test_store_creates_new_currency()
    {
        // بيانات العملة الجديدة
        $currencyData = [
            'name' => 'الجنيه الإسترليني',
            'code' => 'GBP',
            'symbol' => '£',
            'country' => 'المملكة المتحدة',
            'is_active' => true,
            'decimal_places' => 2,
            'sort_order' => 3
        ];

        // إرسال طلب إنشاء عملة جديدة
        $response = $this->post(route('currencies.store'), $currencyData);

        // التحقق من إعادة التوجيه بعد الإنشاء
        $response->assertRedirect(route('currencies.index'));

        // التحقق من وجود العملة في قاعدة البيانات
        $this->assertDatabaseHas('currencies', [
            'name' => 'الجنيه الإسترليني',
            'code' => 'GBP'
        ]);
    }

    /**
     * اختبار عرض تفاصيل عملة
     */
    public function test_show_displays_currency_details()
    {
        // إنشاء عملة للاختبار
        $currency = Currency::create([
            'name' => 'الين الياباني',
            'code' => 'JPY',
            'symbol' => '¥',
            'country' => 'اليابان',
            'is_active' => true,
            'decimal_places' => 0,
            'sort_order' => 4
        ]);

        // زيارة صفحة تفاصيل العملة
        $response = $this->get(route('currencies.show', $currency->id));

        // التحقق من نجاح الاستجابة
        $response->assertStatus(200);

        // التحقق من وجود تفاصيل العملة في الصفحة
        $response->assertSee('الين الياباني');
        $response->assertSee('JPY');
        $response->assertSee('اليابان');
    }

    /**
     * اختبار تحديث عملة
     */
    public function test_update_modifies_currency()
    {
        // إنشاء عملة للاختبار
        $currency = Currency::create([
            'name' => 'الفرنك السويسري',
            'code' => 'CHF',
            'symbol' => 'Fr',
            'country' => 'سويسرا',
            'is_active' => true,
            'decimal_places' => 2,
            'sort_order' => 5
        ]);

        // بيانات التحديث
        $updatedData = [
            'name' => 'الفرنك السويسري المعدل',
            'code' => 'CHF',
            'symbol' => 'CHF',
            'country' => 'سويسرا',
            'is_active' => true,
            'decimal_places' => 2,
            'sort_order' => 5
        ];

        // إرسال طلب تحديث العملة
        $response = $this->put(route('currencies.update', $currency->id), $updatedData);

        // التحقق من إعادة التوجيه بعد التحديث
        $response->assertRedirect(route('currencies.index'));

        // التحقق من تحديث العملة في قاعدة البيانات
        $this->assertDatabaseHas('currencies', [
            'id' => $currency->id,
            'name' => 'الفرنك السويسري المعدل',
            'symbol' => 'CHF'
        ]);
    }

    /**
     * اختبار حذف عملة
     */
    public function test_destroy_deletes_currency()
    {
        // إنشاء عملة للاختبار
        $currency = Currency::create([
            'name' => 'الدولار الكندي',
            'code' => 'CAD',
            'symbol' => 'C$',
            'country' => 'كندا',
            'is_active' => true,
            'decimal_places' => 2,
            'sort_order' => 6
        ]);

        // إرسال طلب حذف العملة
        $response = $this->delete(route('currencies.destroy', $currency->id));

        // التحقق من إعادة التوجيه بعد الحذف
        $response->assertRedirect(route('currencies.index'));

        // التحقق من حذف العملة من قاعدة البيانات
        $this->assertDatabaseMissing('currencies', [
            'id' => $currency->id
        ]);
    }

    /**
     * اختبار تغيير حالة العملة
     */
    public function test_toggle_status_changes_currency_status()
    {
        // إنشاء عملة للاختبار
        $currency = Currency::create([
            'name' => 'الدولار الأسترالي',
            'code' => 'AUD',
            'symbol' => 'A$',
            'country' => 'أستراليا',
            'is_active' => true,
            'decimal_places' => 2,
            'sort_order' => 7
        ]);

        // التحقق من الحالة الأولية
        $this->assertTrue($currency->is_active);

        // إرسال طلب تغيير الحالة
        $response = $this->post(route('currencies.toggle-status', $currency->id));

        // التحقق من إعادة التوجيه بعد تغيير الحالة
        $response->assertRedirect(route('currencies.index'));

        // التحقق من تغيير الحالة في قاعدة البيانات
        $this->assertDatabaseHas('currencies', [
            'id' => $currency->id,
            'is_active' => false
        ]);

        // إرسال طلب تغيير الحالة مرة أخرى
        $response = $this->post(route('currencies.toggle-status', $currency->id));

        // التحقق من تغيير الحالة مرة أخرى في قاعدة البيانات
        $this->assertDatabaseHas('currencies', [
            'id' => $currency->id,
            'is_active' => true
        ]);
    }

    /**
     * اختبار البحث عن العملات
     */
    public function test_search_finds_currencies()
    {
        // إنشاء عملات للاختبار
        Currency::create([
            'name' => 'الريال السعودي',
            'code' => 'SAR',
            'symbol' => 'ر.س',
            'country' => 'المملكة العربية السعودية',
            'is_active' => true,
            'decimal_places' => 2,
            'sort_order' => 8
        ]);

        Currency::create([
            'name' => 'الدرهم الإماراتي',
            'code' => 'AED',
            'symbol' => 'د.إ',
            'country' => 'الإمارات العربية المتحدة',
            'is_active' => true,
            'decimal_places' => 2,
            'sort_order' => 9
        ]);

        // إرسال طلب بحث
        $response = $this->post(route('currencies.search'), [
            'query' => 'الريال'
        ]);

        // التحقق من نجاح الاستجابة
        $response->assertStatus(200);

        // التحقق من وجود العملة المطلوبة في نتائج البحث
        $responseData = $response->json();
        $this->assertCount(1, $responseData);
        $this->assertEquals('الريال السعودي', $responseData[0]['name']);
        $this->assertEquals('SAR', $responseData[0]['code']);

        // إرسال طلب بحث آخر
        $response = $this->post(route('currencies.search'), [
            'query' => 'العربية'
        ]);

        // التحقق من وجود العملات المطلوبة في نتائج البحث
        $responseData = $response->json();
        $this->assertCount(2, $responseData);
    }
}
