<?php
/**
 * CodePlus - نظام الصرافة والحوالات المالية
 * ملف التثبيت الرئيسي
 * 
 * هذا الملف يقوم بتثبيت نظام CodePlus للصرافة والحوالات المالية بطريقة بسيطة
 * مشابهة لسكريبتات Codecanyon
 */

// منع الوصول المباشر للملف إلا من خلال المتصفح
if (PHP_SAPI === 'cli') {
    die("يجب تشغيل هذا الملف من خلال المتصفح.\n");
}

// بدء الجلسة
session_start();

// تعريف الثوابت
define('INSTALLER_VERSION', '1.0');
define('APP_NAME', 'CodePlus');
define('APP_ROOT', dirname(__DIR__));

// تحديد الخطوة الحالية
$step = isset($_GET['step']) ? (int)$_GET['step'] : 1;
if ($step < 1 || $step > 5) {
    $step = 1;
}

// تعريف متغيرات الأخطاء والنجاح
$errors = [];
$success = [];

// معالجة النموذج المرسل
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    switch ($step) {
        case 1:
            // التحقق من المتطلبات - تخطي التحقق ومتابعة للخطوة التالية
            header('Location: install.php?step=2');
            exit;
            break;
            
        case 2:
            // معالجة معلومات قاعدة البيانات
            if (isset($_POST['db_host'], $_POST['db_name'], $_POST['db_user'], $_POST['db_pass'])) {
                $dbHost = $_POST['db_host'];
                $dbName = $_POST['db_name'];
                $dbUser = $_POST['db_user'];
                $dbPass = $_POST['db_pass'];
                
                // اختبار الاتصال بقاعدة البيانات
                try {
                    $dsn = "mysql:host=$dbHost;charset=utf8mb4";
                    $options = [
                        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                        PDO::ATTR_EMULATE_PREPARES => false,
                    ];
                    $pdo = new PDO($dsn, $dbUser, $dbPass, $options);
                    
                    // التحقق من وجود قاعدة البيانات
                    $stmt = $pdo->query("SELECT SCHEMA_NAME FROM INFORMATION_SCHEMA.SCHEMATA WHERE SCHEMA_NAME = '$dbName'");
                    $dbExists = $stmt->fetchColumn();
                    
                    if (!$dbExists) {
                        // إنشاء قاعدة البيانات إذا لم تكن موجودة
                        $pdo->exec("CREATE DATABASE `$dbName` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
                        $success[] = "تم إنشاء قاعدة البيانات '$dbName' بنجاح.";
                    }
                    
                    // حفظ معلومات قاعدة البيانات في ملف الجلسة
                    $_SESSION['db_config'] = [
                        'host' => $dbHost,
                        'name' => $dbName,
                        'user' => $dbUser,
                        'pass' => $dbPass
                    ];
                    
                    header('Location: install.php?step=3');
                    exit;
                } catch (PDOException $e) {
                    $errors[] = "خطأ في الاتصال بقاعدة البيانات: " . $e->getMessage();
                }
            }
            break;
            
        case 3:
            // معالجة معلومات الموقع
            if (isset($_POST['app_name'], $_POST['app_url'], $_POST['admin_email'], $_POST['admin_password'])) {
                $_SESSION['app_config'] = [
                    'name' => $_POST['app_name'],
                    'url' => $_POST['app_url'],
                    'admin_email' => $_POST['admin_email'],
                    'admin_password' => $_POST['admin_password']
                ];
                
                header('Location: install.php?step=4');
                exit;
            }
            break;
            
        case 4:
            // تثبيت النظام
            if (isset($_POST['install'])) {
                $result = installSystem();
                if ($result) {
                    header('Location: install.php?step=5');
                    exit;
                }
            }
            break;
    }
}

/**
 * التحقق من متطلبات النظام
 */
function checkRequirements() {
    global $errors;
    
    $requirements = [
        'php' => '8.0.0',
        'extensions' => [
            'BCMath',
            'Ctype',
            'Fileinfo',
            'JSON',
            'Mbstring',
            'OpenSSL',
            'PDO',
            'Tokenizer',
            'XML'
        ],
        'functions' => [
            'exec',
            'system',
            'shell_exec'
        ],
        'writable' => [
            'storage/',
            'bootstrap/cache/',
            '.env.example'
        ]
    ];
    
    $allPassed = true;
    
    // التحقق من إصدار PHP
    if (version_compare(PHP_VERSION, $requirements['php'], '<')) {
        $errors[] = "إصدار PHP المطلوب هو {$requirements['php']} أو أعلى. الإصدار الحالي هو " . PHP_VERSION;
        $allPassed = false;
    }
    
    // التحقق من امتدادات PHP
    foreach ($requirements['extensions'] as $extension) {
        if (!extension_loaded($extension)) {
            $errors[] = "امتداد PHP المطلوب '$extension' غير مثبت.";
            $allPassed = false;
        }
    }
    
    // التحقق من الدوال المطلوبة
    foreach ($requirements['functions'] as $function) {
        if (!function_exists($function)) {
            $errors[] = "دالة PHP المطلوبة '$function' غير متاحة.";
            $allPassed = false;
        }
    }
    
    // التحقق من صلاحيات الكتابة
    foreach ($requirements['writable'] as $path) {
        $fullPath = APP_ROOT . '/' . $path;
        if (file_exists($fullPath) && !is_writable($fullPath)) {
            $errors[] = "المسار '$path' يجب أن يكون قابلاً للكتابة.";
            $allPassed = false;
        }
    }
    
    return $allPassed;
}

/**
 * تثبيت النظام
 */
function installSystem() {
    global $errors, $success;
    
    try {
        // 1. إنشاء ملف .env
        $envFile = APP_ROOT . '/.env';
        $envExampleFile = APP_ROOT . '/.env.example';
        
        if (file_exists($envExampleFile) && !copy($envExampleFile, $envFile)) {
            throw new Exception("فشل في إنشاء ملف .env");
        }
        
        // 2. تحديث ملف .env بمعلومات قاعدة البيانات والتطبيق
        if (file_exists($envFile)) {
            $envContent = file_get_contents($envFile);
            $dbConfig = $_SESSION['db_config'];
            $appConfig = $_SESSION['app_config'];
            
            $replacements = [
                'APP_NAME=Laravel' => 'APP_NAME="' . $appConfig['name'] . '"',
                'APP_URL=http://localhost' => 'APP_URL=' . $appConfig['url'],
                'DB_HOST=127.0.0.1' => 'DB_HOST=' . $dbConfig['host'],
                'DB_DATABASE=laravel' => 'DB_DATABASE=' . $dbConfig['name'],
                'DB_USERNAME=root' => 'DB_USERNAME=' . $dbConfig['user'],
                'DB_PASSWORD=' => 'DB_PASSWORD=' . $dbConfig['pass']
            ];
            
            foreach ($replacements as $search => $replace) {
                $envContent = str_replace($search, $replace, $envContent);
            }
            
            file_put_contents($envFile, $envContent);
        }
        
        // 3. تنفيذ أوامر Artisan
        $commands = [
            'php artisan key:generate --force',
            'php artisan migrate --force',
            'php artisan db:seed --force',
            'php artisan storage:link'
        ];
        
        foreach ($commands as $command) {
            $output = null;
            $returnVar = null;
            exec('cd ' . APP_ROOT . ' && ' . $command . ' 2>&1', $output, $returnVar);
            
            if ($returnVar !== 0) {
                throw new Exception("فشل في تنفيذ الأمر: $command\n" . implode("\n", $output));
            }
        }
        
        // 4. تحديث بيانات المستخدم الإداري
        $dsn = "mysql:host={$dbConfig['host']};dbname={$dbConfig['name']};charset=utf8mb4";
        $options = [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
        ];
        $pdo = new PDO($dsn, $dbConfig['user'], $dbConfig['pass'], $options);
        
        $hashedPassword = password_hash($appConfig['admin_password'], PASSWORD_BCRYPT);
        
        $stmt = $pdo->prepare("UPDATE users SET email = ?, password = ? WHERE id = 1");
        $stmt->execute([$appConfig['admin_email'], $hashedPassword]);
        
        // 5. إنشاء ملف تثبيت مكتمل
        file_put_contents(APP_ROOT . '/storage/installed', date('Y-m-d H:i:s'));
        
        $success[] = "تم تثبيت النظام بنجاح!";
        return true;
        
    } catch (Exception $e) {
        $errors[] = "حدث خطأ أثناء التثبيت: " . $e->getMessage();
        return false;
    }
}

// عرض رأس الصفحة
function renderHeader() {
    ?>
    <!DOCTYPE html>
    <html dir="rtl" lang="ar">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>تثبيت نظام <?php echo APP_NAME; ?> للصرافة والحوالات المالية</title>
        <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.rtl.min.css">
        <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css">
        <link rel="stylesheet" href="assets/style.css">
    </head>
    <body>
        <div class="installer-container">
            <div class="installer-header">
                <h2>تثبيت نظام <?php echo APP_NAME; ?> للصرافة والحوالات المالية</h2>
                <p>الإصدار <?php echo INSTALLER_VERSION; ?></p>
            </div>
            
            <?php 
                $step = isset($_GET['step']) ? (int)$_GET['step'] : 1;
                if ($step < 1 || $step > 5) {
                    $step = 1;
                }
            ?>

            <div class="installer-body">
                <div class="step-indicator">
                    <?php for ($i = 1; $i <= 5; $i++): ?>
                        <div class="step <?php echo $i < $step ? 'completed' : ($i == $step ? 'active' : ''); ?>">
                            <?php echo $i < $step ? '<i class="bi bi-check"></i>' : $i; ?>
                            <span class="step-label">
                                <?php
                                switch ($i) {
                                    case 1: echo 'المتطلبات'; break;
                                    case 2: echo 'قاعدة البيانات'; break;
                                    case 3: echo 'إعدادات الموقع'; break;
                                    case 4: echo 'التثبيت'; break;
                                    case 5: echo 'اكتمال'; break;
                                }
                                ?>
                            </span>
                        </div>
                    <?php endfor; ?>
                </div>
                
                <?php
                // عرض الأخطاء ورسائل النجاح
                if (!empty($errors)) {
                    echo '<div class="alert alert-danger">';
                    echo '<h5><i class="bi bi-exclamation-triangle"></i> حدثت أخطاء:</h5>';
                    echo '<ul>';
                    foreach ($errors as $error) {
                        echo "<li>$error</li>";
                    }
                    echo '</ul>';
                    echo '</div>';
                }
                
                if (!empty($success)) {
                    echo '<div class="alert alert-success">';
                    echo '<h5><i class="bi bi-check-circle"></i> تم بنجاح:</h5>';
                    echo '<ul>';
                    foreach ($success as $message) {
                        echo "<li>$message</li>";
                    }
                    echo '</ul>';
                    echo '</div>';
                }
                ?>
    <?php
}

// عرض تذييل الصفحة
function renderFooter() {
    ?>
            </div>
            
            <div class="installer-footer">
                <p><?php echo APP_NAME; ?> &copy; <?php echo date('Y'); ?> - نظام الصرافة والحوالات المالية</p>
            </div>
        </div>
        
        <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    </body>
    </html>
    <?php
}

// عرض صفحة التحقق من المتطلبات
function renderRequirements() {
    $requirementsMet = true; // تعيين القيمة إلى true دائماً للسماح بالمتابعة
    ?>
    <h3>التحقق من متطلبات النظام</h3>
    <p>يتحقق النظام من توفر المتطلبات التالية قبل المتابعة:</p>
    
    <div class="card mb-4">
        <div class="card-header">
            <h5>إصدار PHP</h5>
        </div>
        <div class="card-body">
            <div class="requirement-item">
                <span>PHP 8.0.0+</span>
                <span class="requirement-status <?php echo version_compare(PHP_VERSION, '8.0.0', '>=') ? 'status-passed' : 'status-failed'; ?>">
                    <?php echo PHP_VERSION; ?> 
                    <?php echo version_compare(PHP_VERSION, '8.0.0', '>=') ? '<i class="bi bi-check-circle"></i>' : '<i class="bi bi-x-circle"></i>'; ?>
                </span>
            </div>
        </div>
    </div>
    
    <div class="card mb-4">
        <div class="card-header">
            <h5>امتدادات PHP المطلوبة</h5>
        </div>
        <div class="card-body">
            <?php foreach (['BCMath', 'Ctype', 'Fileinfo', 'JSON', 'Mbstring', 'OpenSSL', 'PDO', 'Tokenizer', 'XML'] as $extension): ?>
                <div class="requirement-item">
                    <span><?php echo $extension; ?></span>
                    <span class="requirement-status <?php echo extension_loaded($extension) ? 'status-passed' : 'status-failed'; ?>">
                        <?php echo extension_loaded($extension) ? '<i class="bi bi-check-circle"></i> متوفر' : '<i class="bi bi-x-circle"></i> غير متوفر'; ?>
                    </span>
                </div>
            <?php endforeach; ?>
        </div>
    </div>
    
    <div class="card mb-4">
        <div class="card-header">
            <h5>صلاحيات الكتابة</h5>
        </div>
        <div class="card-body">
            <?php foreach (['storage/', 'bootstrap/cache/', '.env.example'] as $path): ?>
                <?php $fullPath = APP_ROOT . '/' . $path; ?>
                <div class="requirement-item">
                    <span><?php echo $path; ?></span>
                    <span class="requirement-status <?php echo !file_exists($fullPath) || is_writable($fullPath) ? 'status-passed' : 'status-failed'; ?>">
                        <?php echo !file_exists($fullPath) ? '<i class="bi bi-check-circle"></i> غير موجود (سيتم إنشاؤه)' : (is_writable($fullPath) ? '<i class="bi bi-check-circle"></i> قابل للكتابة' : '<i class="bi bi-x-circle"></i> غير قابل للكتابة'); ?>
                    </span>
                </div>
            <?php endforeach; ?>
        </div>
    </div>
    
    <form method="post" action="?step=1">
        <div class="d-grid">
            <button type="submit" class="btn btn-primary btn-lg">
                متابعة <i class="bi bi-arrow-left"></i>
            </button>
        </div>
    </form>
    <?php
}

// عرض نموذج إعدادات قاعدة البيانات
function renderDatabaseForm() {
    ?>
    <h3>إعدادات قاعدة البيانات</h3>
    <p>أدخل معلومات قاعدة البيانات الخاصة بك:</p>
    
    <form method="post" action="?step=2">
        <div class="mb-3">
            <label for="db_host" class="form-label">مضيف قاعدة البيانات</label>
            <input type="text" class="form-control" id="db_host" name="db_host" value="localhost" required>
            <div class="form-text">عادة ما يكون "localhost" أو "127.0.0.1"</div>
        </div>
        
        <div class="mb-3">
            <label for="db_name" class="form-label">اسم قاعدة البيانات</label>
            <input type="text" class="form-control" id="db_name" name="db_name" value="codeplus_db" required>
            <div class="form-text">سيتم إنشاء قاعدة البيانات إذا لم تكن موجودة</div>
        </div>
        
        <div class="mb-3">
            <label for="db_user" class="form-label">اسم مستخدم قاعدة البيانات</label>
            <input type="text" class="form-control" id="db_user" name="db_user" value="root" required>
        </div>
        
        <div class="mb-3">
            <label for="db_pass" class="form-label">كلمة مرور قاعدة البيانات</label>
            <input type="password" class="form-control" id="db_pass" name="db_pass">
        </div>
        
        <div class="d-flex justify-content-between">
            <a href="?step=1" class="btn btn-secondary">
                <i class="bi bi-arrow-right"></i> السابق
            </a>
            <button type="submit" class="btn btn-primary">
                التالي <i class="bi bi-arrow-left"></i>
            </button>
        </div>
    </form>
    <?php
}

// عرض نموذج إعدادات الموقع
function renderSiteForm() {
    ?>
    <h3>إعدادات الموقع</h3>
    <p>أدخل المعلومات الأساسية للنظام:</p>
    
    <form method="post" action="?step=3">
        <div class="mb-3">
            <label for="app_name" class="form-label">اسم التطبيق</label>
            <input type="text" class="form-control" id="app_name" name="app_name" value="CodePlus" required>
        </div>
        
        <div class="mb-3">
            <label for="app_url" class="form-label">رابط الموقع</label>
            <input type="url" class="form-control" id="app_url" name="app_url" value="http://<?php echo $_SERVER['HTTP_HOST']; ?>" required>
            <div class="form-text">أدخل الرابط الكامل بدون شرطة مائلة في النهاية</div>
        </div>
        
        <div class="mb-3">
            <label for="admin_email" class="form-label">البريد الإلكتروني للمدير</label>
            <input type="email" class="form-control" id="admin_email" name="admin_email" value="admin@codeplus.com" required>
        </div>
        
        <div class="mb-3">
            <label for="admin_password" class="form-label">كلمة مرور المدير</label>
            <input type="password" class="form-control" id="admin_password" name="admin_password" value="password" required>
            <div class="form-text">استخدم كلمة مرور قوية تحتوي على أحرف وأرقام ورموز</div>
        </div>
        
        <div class="d-flex justify-content-between">
            <a href="?step=2" class="btn btn-secondary">
                <i class="bi bi-arrow-right"></i> السابق
            </a>
            <button type="submit" class="btn btn-primary">
                التالي <i class="bi bi-arrow-left"></i>
            </button>
        </div>
    </form>
    <?php
}

// عرض صفحة التثبيت
function renderInstallation() {
    ?>
    <h3>تثبيت النظام</h3>
    <p>جاهز لتثبيت نظام <?php echo APP_NAME; ?> للصرافة والحوالات المالية. انقر على زر "تثبيت" للبدء.</p>
    
    <div class="alert alert-info">
        <i class="bi bi-info-circle"></i> سيقوم النظام بتنفيذ الخطوات التالية:
        <ul>
            <li>إنشاء ملف الإعدادات (.env)</li>
            <li>إنشاء مفتاح التطبيق</li>
            <li>إنشاء جداول قاعدة البيانات</li>
            <li>زرع البيانات الأولية</li>
            <li>إنشاء رابط رمزي لمجلد التخزين</li>
            <li>تحديث بيانات المستخدم الإداري</li>
        </ul>
    </div>
    
    <div class="alert alert-warning">
        <i class="bi bi-exclamation-triangle"></i> قد تستغرق عملية التثبيت بضع دقائق. يرجى عدم إغلاق المتصفح أو تحديث الصفحة.
    </div>
    
    <form method="post" action="?step=4">
        <input type="hidden" name="install" value="1">
        
        <div class="d-flex justify-content-between">
            <a href="?step=3" class="btn btn-secondary">
                <i class="bi bi-arrow-right"></i> السابق
            </a>
            <button type="submit" class="btn btn-success btn-lg">
                <i class="bi bi-download"></i> تثبيت النظام
            </button>
        </div>
    </form>
    <?php
}

// عرض صفحة اكتمال التثبيت
function renderComplete() {
    ?>
    <div class="text-center mb-4">
        <i class="bi bi-check-circle-fill text-success" style="font-size: 80px;"></i>
        <h3 class="mt-3">تم تثبيت النظام بنجاح!</h3>
        <p>تهانينا! تم تثبيت نظام <?php echo APP_NAME; ?> للصرافة والحوالات المالية بنجاح.</p>
    </div>
    
    <div class="alert alert-info">
        <h5><i class="bi bi-info-circle"></i> معلومات تسجيل الدخول:</h5>
        <ul>
            <li><strong>رابط لوحة التحكم:</strong> <a href="<?php echo $_SESSION['app_config']['url']; ?>/login" target="_blank"><?php echo $_SESSION['app_config']['url']; ?>/login</a></li>
            <li><strong>البريد الإلكتروني:</strong> <?php echo $_SESSION['app_config']['admin_email']; ?></li>
            <li><strong>كلمة المرور:</strong> <?php echo str_repeat('*', strlen($_SESSION['app_config']['admin_password'])); ?></li>
        </ul>
    </div>
    
    <div class="alert alert-warning">
        <i class="bi bi-shield-lock"></i> <strong>ملاحظة أمنية هامة:</strong> لأسباب أمنية، يرجى حذف مجلد التثبيت "install" بعد الانتهاء من عملية التثبيت.
    </div>
    
    <div class="d-grid gap-2">
        <a href="<?php echo $_SESSION['app_config']['url']; ?>" class="btn btn-primary btn-lg">
            <i class="bi bi-house"></i> الذهاب إلى الصفحة الرئيسية
        </a>
        <a href="<?php echo $_SESSION['app_config']['url']; ?>/login" class="btn btn-success btn-lg">
            <i class="bi bi-box-arrow-in-right"></i> تسجيل الدخول إلى لوحة التحكم
        </a>
    </div>
    <?php
    
    // حذف بيانات الجلسة بعد اكتمال التثبيت
    unset($_SESSION['db_config']);
    unset($_SESSION['app_config']);
}

// عرض الصفحة المناسبة حسب الخطوة الحالية
renderHeader();

switch ($step) {
    case 1:
        renderRequirements();
        break;
    case 2:
        renderDatabaseForm();
        break;
    case 3:
        renderSiteForm();
        break;
    case 4:
        renderInstallation();
        break;
    case 5:
        renderComplete();
        break;
}

renderFooter();
