#!/bin/bash

# سكريبت تثبيت نظام CodePlus للصرافة والحوالات المالية
# هذا السكريبت يقوم بتثبيت نظام CodePlus بشكل تلقائي وسهل

# تلوين النصوص للتوضيح
GREEN='\033[0;32m'
BLUE='\033[0;34m'
RED='\033[0;31m'
NC='\033[0m' # No Color

# دالة لعرض رسائل النجاح
success() {
    echo -e "${GREEN}$1${NC}"
}

# دالة لعرض رسائل المعلومات
info() {
    echo -e "${BLUE}$1${NC}"
}

# دالة لعرض رسائل الخطأ
error() {
    echo -e "${RED}$1${NC}"
}

# التحقق من وجود الأدوات الضرورية
check_requirements() {
    info "جاري التحقق من المتطلبات الأساسية..."
    
    # التحقق من وجود PHP
    if ! command -v php &> /dev/null; then
        error "PHP غير مثبت. الرجاء تثبيت PHP 8.0 أو أحدث."
        exit 1
    fi
    
    # التحقق من وجود Composer
    if ! command -v composer &> /dev/null; then
        error "Composer غير مثبت. الرجاء تثبيت Composer."
        exit 1
    fi
    
    # التحقق من وجود MySQL
    if ! command -v mysql &> /dev/null; then
        error "MySQL غير مثبت. الرجاء تثبيت MySQL 5.7 أو أحدث."
        exit 1
    fi
    
    success "تم التحقق من المتطلبات الأساسية بنجاح!"
}

# فك ضغط الملف
extract_files() {
    info "جاري فك ضغط ملفات النظام..."
    
    # التحقق من وجود الملف المضغوط
    if [ ! -f "codeplus_currency_exchange_system.zip" ]; then
        error "ملف codeplus_currency_exchange_system.zip غير موجود!"
        exit 1
    fi
    
    # فك ضغط الملف
    unzip -q codeplus_currency_exchange_system.zip
    
    if [ $? -ne 0 ]; then
        error "حدث خطأ أثناء فك ضغط الملفات!"
        exit 1
    fi
    
    success "تم فك ضغط ملفات النظام بنجاح!"
}

# إعداد قاعدة البيانات
setup_database() {
    info "جاري إعداد قاعدة البيانات..."
    
    # طلب معلومات قاعدة البيانات من المستخدم
    read -p "أدخل اسم قاعدة البيانات (الافتراضي: codeplus_db): " DB_NAME
    DB_NAME=${DB_NAME:-codeplus_db}
    
    read -p "أدخل اسم مستخدم قاعدة البيانات (الافتراضي: root): " DB_USER
    DB_USER=${DB_USER:-root}
    
    read -sp "أدخل كلمة مرور قاعدة البيانات: " DB_PASS
    echo ""
    
    read -p "أدخل مضيف قاعدة البيانات (الافتراضي: localhost): " DB_HOST
    DB_HOST=${DB_HOST:-localhost}
    
    # إنشاء قاعدة البيانات
    mysql -h $DB_HOST -u $DB_USER -p$DB_PASS -e "CREATE DATABASE IF NOT EXISTS $DB_NAME CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;"
    
    if [ $? -ne 0 ]; then
        error "حدث خطأ أثناء إنشاء قاعدة البيانات!"
        exit 1
    fi
    
    success "تم إعداد قاعدة البيانات بنجاح!"
}

# تكوين ملف البيئة
configure_env() {
    info "جاري تكوين ملف البيئة..."
    
    cd currency_exchange_app
    
    # نسخ ملف البيئة النموذجي
    cp .env.example .env
    
    if [ $? -ne 0 ]; then
        error "حدث خطأ أثناء نسخ ملف البيئة!"
        exit 1
    fi
    
    # تعديل ملف البيئة
    sed -i "s/DB_DATABASE=laravel/DB_DATABASE=$DB_NAME/" .env
    sed -i "s/DB_USERNAME=root/DB_USERNAME=$DB_USER/" .env
    sed -i "s/DB_PASSWORD=/DB_PASSWORD=$DB_PASS/" .env
    sed -i "s/DB_HOST=127.0.0.1/DB_HOST=$DB_HOST/" .env
    
    success "تم تكوين ملف البيئة بنجاح!"
}

# تثبيت اعتماديات Composer
install_dependencies() {
    info "جاري تثبيت اعتماديات Composer..."
    
    composer install --no-interaction
    
    if [ $? -ne 0 ]; then
        error "حدث خطأ أثناء تثبيت اعتماديات Composer!"
        exit 1
    fi
    
    success "تم تثبيت اعتماديات Composer بنجاح!"
}

# إنشاء مفتاح التطبيق
generate_app_key() {
    info "جاري إنشاء مفتاح التطبيق..."
    
    php artisan key:generate
    
    if [ $? -ne 0 ]; then
        error "حدث خطأ أثناء إنشاء مفتاح التطبيق!"
        exit 1
    fi
    
    success "تم إنشاء مفتاح التطبيق بنجاح!"
}

# تنفيذ هجرات قاعدة البيانات
run_migrations() {
    info "جاري تنفيذ هجرات قاعدة البيانات..."
    
    php artisan migrate --force
    
    if [ $? -ne 0 ]; then
        error "حدث خطأ أثناء تنفيذ هجرات قاعدة البيانات!"
        exit 1
    fi
    
    success "تم تنفيذ هجرات قاعدة البيانات بنجاح!"
}

# زرع البيانات الأولية
seed_database() {
    info "جاري زرع البيانات الأولية..."
    
    php artisan db:seed --force
    
    if [ $? -ne 0 ]; then
        error "حدث خطأ أثناء زرع البيانات الأولية!"
        exit 1
    fi
    
    success "تم زرع البيانات الأولية بنجاح!"
}

# تكوين مجلد التخزين
setup_storage() {
    info "جاري تكوين مجلد التخزين..."
    
    php artisan storage:link
    
    if [ $? -ne 0 ]; then
        error "حدث خطأ أثناء تكوين مجلد التخزين!"
        exit 1
    fi
    
    # ضبط صلاحيات الملفات
    chmod -R 775 storage bootstrap/cache
    
    success "تم تكوين مجلد التخزين بنجاح!"
}

# تشغيل التطبيق
run_application() {
    info "جاري تشغيل التطبيق..."
    
    # عرض معلومات تسجيل الدخول
    echo ""
    echo "==================================================="
    echo "تم تثبيت نظام CodePlus للصرافة والحوالات المالية بنجاح!"
    echo "==================================================="
    echo ""
    echo "معلومات تسجيل الدخول:"
    echo "البريد الإلكتروني: admin@codeplus.com"
    echo "كلمة المرور: password"
    echo ""
    echo "يرجى تغيير كلمة المرور الافتراضية فور تسجيل الدخول لأول مرة."
    echo ""
    
    # سؤال المستخدم إذا كان يريد تشغيل خادم التطوير
    read -p "هل تريد تشغيل خادم التطوير الآن؟ (y/n): " RUN_SERVER
    
    if [[ $RUN_SERVER == "y" || $RUN_SERVER == "Y" ]]; then
        php artisan serve
    else
        echo ""
        echo "لتشغيل التطبيق لاحقاً، قم بتنفيذ الأمر التالي:"
        echo "cd $(pwd) && php artisan serve"
        echo ""
        echo "ثم قم بزيارة http://localhost:8000 في المتصفح."
    fi
}

# الدالة الرئيسية
main() {
    echo "==================================================="
    echo "  سكريبت تثبيت نظام CodePlus للصرافة والحوالات المالية  "
    echo "==================================================="
    echo ""
    
    # تنفيذ خطوات التثبيت
    check_requirements
    extract_files
    setup_database
    configure_env
    install_dependencies
    generate_app_key
    run_migrations
    seed_database
    setup_storage
    run_application
}

# تنفيذ الدالة الرئيسية
main
