<?php

namespace App\Services;

use App\Core\SystemCore;
use App\Helpers\CurrencyHelper;

/**
 * خدمة إدارة العملات وأسعار الصرف
 */
class CurrencyService
{
    /**
     * نواة النظام
     */
    private $core;
    
    /**
     * اتصال قاعدة البيانات
     */
    private $db;
    
    /**
     * إنشاء نسخة جديدة من خدمة العملات
     * 
     * @param SystemCore $core نواة النظام
     * @param \PDO $db اتصال قاعدة البيانات
     */
    public function __construct(SystemCore $core, \PDO $db)
    {
        $this->core = $core;
        $this->db = $db;
    }
    
    /**
     * الحصول على جميع العملات
     * 
     * @return array قائمة العملات
     */
    public function getAllCurrencies()
    {
        $stmt = $this->db->prepare("SELECT * FROM currencies ORDER BY name");
        $stmt->execute();
        return $stmt->fetchAll(\PDO::FETCH_ASSOC);
    }
    
    /**
     * الحصول على عملة محددة
     * 
     * @param int $id معرف العملة
     * @return array|null بيانات العملة
     */
    public function getCurrency($id)
    {
        $stmt = $this->db->prepare("SELECT * FROM currencies WHERE id = :id");
        $stmt->execute(['id' => $id]);
        return $stmt->fetch(\PDO::FETCH_ASSOC);
    }
    
    /**
     * الحصول على عملة بواسطة الرمز
     * 
     * @param string $code رمز العملة
     * @return array|null بيانات العملة
     */
    public function getCurrencyByCode($code)
    {
        $stmt = $this->db->prepare("SELECT * FROM currencies WHERE code = :code");
        $stmt->execute(['code' => $code]);
        return $stmt->fetch(\PDO::FETCH_ASSOC);
    }
    
    /**
     * إضافة عملة جديدة
     * 
     * @param array $data بيانات العملة
     * @return bool|int نتيجة الإضافة
     */
    public function addCurrency($data)
    {
        // التحقق من وجود العملة
        $stmt = $this->db->prepare("SELECT COUNT(*) FROM currencies WHERE code = :code");
        $stmt->execute(['code' => $data['code']]);
        
        if ($stmt->fetchColumn() > 0) {
            return false;
        }
        
        $stmt = $this->db->prepare("
            INSERT INTO currencies (
                name, code, symbol, is_active, created_at, updated_at
            ) VALUES (
                :name, :code, :symbol, :is_active, NOW(), NOW()
            )
        ");
        
        $result = $stmt->execute([
            'name' => $data['name'],
            'code' => $data['code'],
            'symbol' => $data['symbol'],
            'is_active' => $data['is_active'] ?? true
        ]);
        
        if ($result) {
            return $this->db->lastInsertId();
        }
        
        return false;
    }
    
    /**
     * تحديث بيانات عملة
     * 
     * @param int $id معرف العملة
     * @param array $data بيانات العملة
     * @return bool نتيجة التحديث
     */
    public function updateCurrency($id, $data)
    {
        // التحقق من وجود العملة
        $stmt = $this->db->prepare("SELECT COUNT(*) FROM currencies WHERE code = :code AND id != :id");
        $stmt->execute([
            'code' => $data['code'],
            'id' => $id
        ]);
        
        if ($stmt->fetchColumn() > 0) {
            return false;
        }
        
        $stmt = $this->db->prepare("
            UPDATE currencies SET
                name = :name,
                code = :code,
                symbol = :symbol,
                is_active = :is_active,
                updated_at = NOW()
            WHERE id = :id
        ");
        
        return $stmt->execute([
            'id' => $id,
            'name' => $data['name'],
            'code' => $data['code'],
            'symbol' => $data['symbol'],
            'is_active' => $data['is_active'] ?? true
        ]);
    }
    
    /**
     * حذف عملة
     * 
     * @param int $id معرف العملة
     * @return bool نتيجة الحذف
     */
    public function deleteCurrency($id)
    {
        // التحقق من استخدام العملة في أسعار الصرف
        $stmt = $this->db->prepare("
            SELECT COUNT(*) FROM exchange_rates 
            WHERE from_currency_id = :id OR to_currency_id = :id
        ");
        $stmt->execute(['id' => $id]);
        
        if ($stmt->fetchColumn() > 0) {
            return false;
        }
        
        $stmt = $this->db->prepare("DELETE FROM currencies WHERE id = :id");
        return $stmt->execute(['id' => $id]);
    }
    
    /**
     * الحصول على جميع أسعار الصرف
     * 
     * @return array قائمة أسعار الصرف
     */
    public function getAllExchangeRates()
    {
        $stmt = $this->db->prepare("
            SELECT er.*, 
                fc.code as from_currency_code, fc.name as from_currency_name,
                tc.code as to_currency_code, tc.name as to_currency_name
            FROM exchange_rates er
            JOIN currencies fc ON er.from_currency_id = fc.id
            JOIN currencies tc ON er.to_currency_id = tc.id
            ORDER BY fc.name, tc.name
        ");
        $stmt->execute();
        return $stmt->fetchAll(\PDO::FETCH_ASSOC);
    }
    
    /**
     * الحصول على سعر صرف محدد
     * 
     * @param int $id معرف سعر الصرف
     * @return array|null بيانات سعر الصرف
     */
    public function getExchangeRate($id)
    {
        $stmt = $this->db->prepare("
            SELECT er.*, 
                fc.code as from_currency_code, fc.name as from_currency_name,
                tc.code as to_currency_code, tc.name as to_currency_name
            FROM exchange_rates er
            JOIN currencies fc ON er.from_currency_id = fc.id
            JOIN currencies tc ON er.to_currency_id = tc.id
            WHERE er.id = :id
        ");
        $stmt->execute(['id' => $id]);
        return $stmt->fetch(\PDO::FETCH_ASSOC);
    }
    
    /**
     * الحصول على سعر صرف بين عملتين
     * 
     * @param int $fromCurrencyId معرف العملة المصدر
     * @param int $toCurrencyId معرف العملة الهدف
     * @return array|null بيانات سعر الصرف
     */
    public function getExchangeRateByCurrencies($fromCurrencyId, $toCurrencyId)
    {
        $stmt = $this->db->prepare("
            SELECT * FROM exchange_rates 
            WHERE from_currency_id = :from_currency_id AND to_currency_id = :to_currency_id
        ");
        $stmt->execute([
            'from_currency_id' => $fromCurrencyId,
            'to_currency_id' => $toCurrencyId
        ]);
        return $stmt->fetch(\PDO::FETCH_ASSOC);
    }
    
    /**
     * إضافة سعر صرف جديد
     * 
     * @param array $data بيانات سعر الصرف
     * @return bool|int نتيجة الإضافة
     */
    public function addExchangeRate($data)
    {
        // التحقق من وجود سعر الصرف
        $stmt = $this->db->prepare("
            SELECT COUNT(*) FROM exchange_rates 
            WHERE from_currency_id = :from_currency_id AND to_currency_id = :to_currency_id
        ");
        $stmt->execute([
            'from_currency_id' => $data['from_currency_id'],
            'to_currency_id' => $data['to_currency_id']
        ]);
        
        if ($stmt->fetchColumn() > 0) {
            return false;
        }
        
        $stmt = $this->db->prepare("
            INSERT INTO exchange_rates (
                from_currency_id, to_currency_id, rate, is_active, created_at, updated_at
            ) VALUES (
                :from_currency_id, :to_currency_id, :rate, :is_active, NOW(), NOW()
            )
        ");
        
        $result = $stmt->execute([
            'from_currency_id' => $data['from_currency_id'],
            'to_currency_id' => $data['to_currency_id'],
            'rate' => $data['rate'],
            'is_active' => $data['is_active'] ?? true
        ]);
        
        if ($result) {
            return $this->db->lastInsertId();
        }
        
        return false;
    }
    
    /**
     * تحديث بيانات سعر صرف
     * 
     * @param int $id معرف سعر الصرف
     * @param array $data بيانات سعر الصرف
     * @return bool نتيجة التحديث
     */
    public function updateExchangeRate($id, $data)
    {
        // التحقق من وجود سعر الصرف
        $stmt = $this->db->prepare("
            SELECT COUNT(*) FROM exchange_rates 
            WHERE from_currency_id = :from_currency_id AND to_currency_id = :to_currency_id AND id != :id
        ");
        $stmt->execute([
            'from_currency_id' => $data['from_currency_id'],
            'to_currency_id' => $data['to_currency_id'],
            'id' => $id
        ]);
        
        if ($stmt->fetchColumn() > 0) {
            return false;
        }
        
        $stmt = $this->db->prepare("
            UPDATE exchange_rates SET
                from_currency_id = :from_currency_id,
                to_currency_id = :to_currency_id,
                rate = :rate,
                is_active = :is_active,
                updated_at = NOW()
            WHERE id = :id
        ");
        
        return $stmt->execute([
            'id' => $id,
            'from_currency_id' => $data['from_currency_id'],
            'to_currency_id' => $data['to_currency_id'],
            'rate' => $data['rate'],
            'is_active' => $data['is_active'] ?? true
        ]);
    }
    
    /**
     * حذف سعر صرف
     * 
     * @param int $id معرف سعر الصرف
     * @return bool نتيجة الحذف
     */
    public function deleteExchangeRate($id)
    {
        $stmt = $this->db->prepare("DELETE FROM exchange_rates WHERE id = :id");
        return $stmt->execute(['id' => $id]);
    }
    
    /**
     * تحويل مبلغ من عملة إلى أخرى
     * 
     * @param float $amount المبلغ
     * @param int $fromCurrencyId معرف العملة المصدر
     * @param int $toCurrencyId معرف العملة الهدف
     * @return float|bool المبلغ المحول أو false في حالة الفشل
     */
    public function convertAmount($amount, $fromCurrencyId, $toCurrencyId)
    {
        if ($fromCurrencyId === $toCurrencyId) {
            return $amount;
        }
        
        // الحصول على العملات
        $fromCurrency = $this->getCurrency($fromCurrencyId);
        $toCurrency = $this->getCurrency($toCurrencyId);
        
        if (!$fromCurrency || !$toCurrency) {
            return false;
        }
        
        // الحصول على سعر الصرف
        $exchangeRate = $this->getExchangeRateByCurrencies($fromCurrencyId, $toCurrencyId);
        
        if ($exchangeRate) {
            return $amount * $exchangeRate['rate'];
        }
        
        // محاولة الحصول على سعر الصرف العكسي
        $reverseExchangeRate = $this->getExchangeRateByCurrencies($toCurrencyId, $fromCurrencyId);
        
        if ($reverseExchangeRate) {
            return $amount / $reverseExchangeRate['rate'];
        }
        
        // محاولة التحويل من خلال عملة وسيطة (الدولار الأمريكي)
        $usdCurrency = $this->getCurrencyByCode('USD');
        
        if (!$usdCurrency) {
            return false;
        }
        
        $fromToUsdRate = $this->getExchangeRateByCurrencies($fromCurrencyId, $usdCurrency['id']);
        $usdToToRate = $this->getExchangeRateByCurrencies($usdCurrency['id'], $toCurrencyId);
        
        if ($fromToUsdRate && $usdToToRate) {
            $amountInUsd = $amount * $fromToUsdRate['rate'];
            return $amountInUsd * $usdToToRate['rate'];
        }
        
        return false;
    }
}
