<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Role extends Model
{
    use HasFactory;

     /**
     * The attributes that aren't mass assignable.
     *
     * @var array
     */
    protected $guarded = ['id'];

    /**
     * علاقة الدور بالمستخدمين.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function users()
    {
        return $this->hasMany(User::class);
    }


    public function permissions(){
        return $this->belongsToMany(Permission::class, 'role_permissions', 'role_id', 'permission_id');
    }

    
    /**
     * التحقق مما إذا كان الدور يملك صلاحية معينة.
     *
     * @param string $permission
     * @return bool
     */
    public function hasPermission($permission)
    {
        return $this->permissions()->where('name', $permission)->exists();
    }

    /**
     * إضافة صلاحية للدور.
     *
     * @param Permission|array|int $permission
     * @return void
     */
    public function givePermissionTo($permission)
    {
        $this->permissions()->syncWithoutDetaching($this->getPermissionId($permission));
    }

    /**
     * إزالة صلاحية من الدور.
     *
     * @param Permission|array|int $permission
     * @return void
     */
    public function revokePermissionTo($permission)
    {
        $this->permissions()->detach($this->getPermissionId($permission));
    }

    /**
     * الحصول على معرف الصلاحية.
     *
     * @param Permission|array|int $permission
     * @return int|array
     */
    protected function getPermissionId($permission)
    {
        if (is_numeric($permission)) {
            return $permission;
        }

        if (is_string($permission)) {
            return Permission::where('name', $permission)->firstOrFail()->id;
        }

        if (is_array($permission)) {
            return Permission::whereIn('name', $permission)->get()->pluck('id')->toArray();
        }

        return $permission->id;
    }
}
