<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Remittance extends Model
{
    use HasFactory;
    
      /**
     * The attributes that aren't mass assignable.
     *
     * @var array
     */
    protected $guarded = ['id'];

    /**
     * الحقول التي يجب تحويلها إلى أنواع أخرى
     */
    protected $casts = [
        'amount' => 'decimal:2',
        'fee' => 'decimal:2',
        'exchange_rate' => 'decimal:6',
        'total_amount' => 'decimal:2',
        'expected_delivery_date' => 'datetime',
        'actual_delivery_date' => 'datetime',
    ];

    /**
     * حالات الحوالة
     */
    const STATUS_PENDING = 'pending';
    const STATUS_PROCESSING = 'processing';
    const STATUS_COMPLETED = 'completed';
    const STATUS_CANCELLED = 'cancelled';
    const STATUS_REFUNDED = 'refunded';
    const STATUS_ON_HOLD = 'on_hold';

    /**
     * طرق الدفع
     */
    const PAYMENT_CASH = 'cash';
    const PAYMENT_BANK_TRANSFER = 'bank_transfer';
    const PAYMENT_CREDIT_CARD = 'credit_card';
    const PAYMENT_MOBILE_WALLET = 'mobile_wallet';

    /**
     * العلاقة مع المرسل (العميل)
     */
    public function sender()
    {
        return $this->belongsTo(Customer::class, 'sender_id');
    }

    /**
     * العلاقة مع العملة المصدر
     */
    public function fromCurrency()
    {
        return $this->belongsTo(Currency::class, 'from_currency_id');
    }

    /**
     * العلاقة مع العملة الهدف
     */
    public function toCurrency()
    {
        return $this->belongsTo(Currency::class, 'to_currency_id');
    }

    /**
     * العلاقة مع المستخدم الذي أنشأ الحوالة
     */
    public function createdBy()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * العلاقة مع المستخدم الذي قام بتحديث الحوالة
     */
    public function updatedBy()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    /**
     * الحصول على الحوالات حسب الحالة
     */
    public function scopeWithStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * الحصول على الحوالات المعلقة
     */
    public function scopePending($query)
    {
        return $query->where('status', self::STATUS_PENDING);
    }

    /**
     * الحصول على الحوالات قيد المعالجة
     */
    public function scopeProcessing($query)
    {
        return $query->where('status', self::STATUS_PROCESSING);
    }

    /**
     * الحصول على الحوالات المكتملة
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', self::STATUS_COMPLETED);
    }

    /**
     * الحصول على الحوالات الملغاة
     */
    public function scopeCancelled($query)
    {
        return $query->where('status', self::STATUS_CANCELLED);
    }

    /**
     * الحصول على الحوالات المستردة
     */
    public function scopeRefunded($query)
    {
        return $query->where('status', self::STATUS_REFUNDED);
    }

    /**
     * الحصول على الحوالات المعلقة مؤقتاً
     */
    public function scopeOnHold($query)
    {
        return $query->where('status', self::STATUS_ON_HOLD);
    }

    /**
     * الحصول على الحوالات حسب بلد الوجهة
     */
    public function scopeForDestinationCountry($query, $country)
    {
        return $query->where('destination_country', $country);
    }

    /**
     * الحصول على الحوالات حسب العميل المرسل
     */
    public function scopeForSender($query, $senderId)
    {
        return $query->where('sender_id', $senderId);
    }

    /**
     * الحصول على الحوالات حسب العملة المصدر
     */
    public function scopeForFromCurrency($query, $currencyId)
    {
        return $query->where('from_currency_id', $currencyId);
    }

    /**
     * الحصول على الحوالات حسب العملة الهدف
     */
    public function scopeForToCurrency($query, $currencyId)
    {
        return $query->where('to_currency_id', $currencyId);
    }

    /**
     * الحصول على الحوالات التي تم إنشاؤها في تاريخ معين
     */
    public function scopeCreatedOn($query, $date)
    {
        return $query->whereDate('created_at', $date);
    }

    /**
     * الحصول على الحوالات التي تم إنشاؤها في فترة زمنية معينة
     */
    public function scopeCreatedBetween($query, $startDate, $endDate)
    {
        return $query->whereBetween('created_at', [$startDate, $endDate]);
    }

    /**
     * توليد رقم مرجعي فريد للحوالة
     */
    public static function generateReferenceNumber()
    {
        $prefix = 'RM';
        $timestamp = now()->format('YmdHis');
        $random = rand(1000, 9999);
        
        return $prefix . $timestamp . $random;
    }

    /**
     * الحصول على اسم حالة الحوالة بالعربية
     */
    public function getStatusNameAttribute()
    {
        $statusNames = [
            self::STATUS_PENDING => 'معلقة',
            self::STATUS_PROCESSING => 'قيد المعالجة',
            self::STATUS_COMPLETED => 'مكتملة',
            self::STATUS_CANCELLED => 'ملغاة',
            self::STATUS_REFUNDED => 'مستردة',
            self::STATUS_ON_HOLD => 'معلقة مؤقتاً'
        ];
        
        return $statusNames[$this->status] ?? $this->status;
    }

    /**
     * الحصول على اسم طريقة الدفع بالعربية
     */
    public function getPaymentMethodNameAttribute()
    {
        $methodNames = [
            self::PAYMENT_CASH => 'نقداً',
            self::PAYMENT_BANK_TRANSFER => 'تحويل بنكي',
            self::PAYMENT_CREDIT_CARD => 'بطاقة ائتمان',
            self::PAYMENT_MOBILE_WALLET => 'محفظة إلكترونية'
        ];
        
        return $methodNames[$this->payment_method] ?? $this->payment_method;
    }

    /**
     * الحصول على المبلغ المحول بعملة الهدف
     */
    public function getConvertedAmountAttribute()
    {
        return $this->amount * $this->exchange_rate;
    }
}
