<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class ExchangeRate extends Model
{
    use HasFactory;
  /**
     * The attributes that aren't mass assignable.
     *
     * @var array
     */
    protected $guarded = ['id'];

    /**
     * الحقول التي يجب تحويلها إلى أنواع أخرى
     */
    protected $casts = [
        'buy_rate' => 'decimal:6',
        'sell_rate' => 'decimal:6',
        'is_active' => 'boolean',
        'effective_date' => 'datetime'
    ];

    /**
     * العلاقة مع العملة المصدر
     */
    public function fromCurrency()
    {
        return $this->belongsTo(Currency::class, 'from_currency_id');
    }

    /**
     * العلاقة مع العملة الهدف
     */
    public function toCurrency()
    {
        return $this->belongsTo(Currency::class, 'to_currency_id');
    }

    /**
     * العلاقة مع المستخدم الذي قام بتحديث السعر
     */
    public function updatedBy()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    /**
     * الحصول على أسعار الصرف النشطة فقط
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * الحصول على أسعار الصرف الفعالة في تاريخ معين
     */
    public function scopeEffectiveOn($query, $date = null)
    {
        $date = $date ?? now();
        return $query->where('effective_date', '<=', $date);
    }

    /**
     * الحصول على أسعار الصرف المرتبة حسب التاريخ الفعال (الأحدث أولاً)
     */
    public function scopeLatest($query)
    {
        return $query->orderBy('effective_date', 'desc');
    }

    /**
     * الحصول على أسعار الصرف لعملة محددة
     */
    public function scopeForCurrency($query, $currencyId)
    {
        return $query->where('from_currency_id', $currencyId)
                     ->orWhere('to_currency_id', $currencyId);
    }

    /**
     * الحصول على أسعار الصرف بين عملتين محددتين
     */
    public function scopeBetweenCurrencies($query, $fromCurrencyId, $toCurrencyId)
    {
        return $query->where(function($q) use ($fromCurrencyId, $toCurrencyId) {
            $q->where('from_currency_id', $fromCurrencyId)
              ->where('to_currency_id', $toCurrencyId);
        })->orWhere(function($q) use ($fromCurrencyId, $toCurrencyId) {
            $q->where('from_currency_id', $toCurrencyId)
              ->where('to_currency_id', $fromCurrencyId);
        });
    }

    /**
     * حساب سعر الصرف العكسي
     */
    public function getInverseBuyRateAttribute()
    {
        return $this->buy_rate > 0 ? (1 / $this->buy_rate) : 0;
    }

    /**
     * حساب سعر الصرف العكسي للبيع
     */
    public function getInverseSellRateAttribute()
    {
        return $this->sell_rate > 0 ? (1 / $this->sell_rate) : 0;
    }

    /**
     * حساب الفرق بين سعر البيع وسعر الشراء
     */
    public function getSpreadAttribute()
    {
        return $this->sell_rate - $this->buy_rate;
    }

    /**
     * حساب نسبة الفرق بين سعر البيع وسعر الشراء
     */
    public function getSpreadPercentAttribute()
    {
        return $this->buy_rate > 0 ? (($this->sell_rate - $this->buy_rate) / $this->buy_rate) * 100 : 0;
    }
}
