<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Currency extends Model
{
    use HasFactory;
    
  /**
     * The attributes that aren't mass assignable.
     *
     * @var array
     */
    protected $guarded = ['id'];

    /**
     * الحقول التي يجب تحويلها إلى أنواع أخرى
     */
    protected $casts = [
        'is_active' => 'boolean',
        'decimal_places' => 'integer',
        'sort_order' => 'integer'
    ];

    /**
     * الحصول على أسعار الصرف التي تكون هذه العملة هي العملة الأساسية
     */
    public function fromExchangeRates()
    {
        return $this->hasMany(ExchangeRate::class, 'from_currency_id');
    }

    /**
     * الحصول على أسعار الصرف التي تكون هذه العملة هي العملة الثانوية
     */
    public function toExchangeRates()
    {
        return $this->hasMany(ExchangeRate::class, 'to_currency_id');
    }

    /**
     * الحصول على المعاملات التي تكون هذه العملة هي العملة المصدر
     */
    public function fromTransactions()
    {
        return $this->hasMany(Transaction::class, 'from_currency_id');
    }

    /**
     * الحصول على المعاملات التي تكون هذه العملة هي العملة الهدف
     */
    public function toTransactions()
    {
        return $this->hasMany(Transaction::class, 'to_currency_id');
    }

    /**
     * الحصول على الحوالات التي تكون هذه العملة هي العملة المصدر
     */
    public function fromRemittances()
    {
        return $this->hasMany(Remittance::class, 'from_currency_id');
    }

    /**
     * الحصول على الحوالات التي تكون هذه العملة هي العملة الهدف
     */
    public function toRemittances()
    {
        return $this->hasMany(Remittance::class, 'to_currency_id');
    }

    /**
     * الحصول على سعر الصرف بين هذه العملة وعملة أخرى
     */
    public function getExchangeRateTo($toCurrencyId)
    {
        return $this->fromExchangeRates()
            ->where('to_currency_id', $toCurrencyId)
            ->where('is_active', true)
            ->first();
    }

    /**
     * الحصول على العملات النشطة فقط
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * الحصول على العملات مرتبة حسب الترتيب
     */
    public function scopeOrdered($query)
    {
        return $query->orderBy('sort_order')->orderBy('code');
    }
}
