<?php

namespace App\Http\Controllers;

use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class SettingsController extends Controller
{
    /**
     * عرض صفحة الإعدادات
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $systemSettings = Setting::getByGroup('system');
        $appearanceSettings = Setting::getByGroup('appearance');
        $businessSettings = Setting::getByGroup('business');
        
        return view('settings.index', compact('systemSettings', 'appearanceSettings', 'businessSettings'));
    }
    
    /**
     * تحديث الإعدادات
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request)
    {
        $settings = $request->except('_token', '_method');
        
        foreach ($settings as $key => $value) {
            Setting::setValue($key, $value);
        }
        
        return redirect()->route('settings.index')->with('success', 'تم تحديث الإعدادات بنجاح');
    }
    
    /**
     * الحصول على إعداد معين
     *
     * @param  string  $key
     * @return \Illuminate\Http\Response
     */
    public function get($key)
    {
        $setting = Setting::where('key', $key)->first();
        
        if (!$setting || !$setting->is_public) {
            return response()->json(['error' => 'الإعداد غير موجود أو غير متاح للعرض'], 404);
        }
        
        return response()->json(['key' => $setting->key, 'value' => $setting->value]);
    }
    
    /**
     * الحصول على جميع الإعدادات العامة
     *
     * @return \Illuminate\Http\Response
     */
    public function getPublicSettings()
    {
        $settings = Setting::getPublicSettings();
        
        $formattedSettings = [];
        foreach ($settings as $setting) {
            $formattedSettings[$setting->key] = $setting->value;
        }
        
        return response()->json($formattedSettings);
    }
    
    /**
     * تحديث شعار النظام
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function updateLogo(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'logo' => 'required|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ]);
        
        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator);
        }
        
        $logoName = 'logo.' . $request->file('logo')->getClientOriginalExtension();
        $request->file('logo')->move(public_path('images'), $logoName);
        
        Setting::setValue('system_logo', $logoName);
        
        return redirect()->route('settings.index')->with('success', 'تم تحديث شعار النظام بنجاح');
    }
}
