<?php

namespace App\Http\Controllers;

use App\Models\Currency;
use App\Models\ExchangeRate;
use App\Models\Remittance;
use App\Models\Customer;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class ReportController extends Controller
{
    /**
     * عرض صفحة التقارير الرئيسية
     */
    public function index()
    {
        return view('reports.index');
    }

    /**
     * تقرير الحوالات المالية
     */
    public function remittancesReport(Request $request)
    {
        // التحقق من المعلمات
        $startDate = $request->input('start_date') ? Carbon::parse($request->input('start_date')) : Carbon::now()->startOfMonth();
        $endDate = $request->input('end_date') ? Carbon::parse($request->input('end_date')) : Carbon::now();
        $status = $request->input('status');
        $destinationCountry = $request->input('destination_country');
        
        // بناء الاستعلام
        $query = Remittance::with(['sender', 'fromCurrency', 'toCurrency'])
            ->whereBetween('created_at', [$startDate->startOfDay(), $endDate->endOfDay()]);
            
        if ($status) {
            $query->where('status', $status);
        }
        
        if ($destinationCountry) {
            $query->where('destination_country', $destinationCountry);
        }
        
        // الحصول على البيانات
        $remittances = $query->orderBy('created_at', 'desc')->get();
        
        // حساب الإحصائيات
        $totalCount = $remittances->count();
        $totalAmount = $remittances->sum('amount');
        $totalFees = $remittances->sum('fee');
        
        // تجميع البيانات حسب العملة
        $byCurrency = $remittances->groupBy('from_currency_id')
            ->map(function ($items, $currencyId) {
                $currency = Currency::find($currencyId);
                return [
                    'currency' => $currency->code,
                    'count' => $items->count(),
                    'amount' => $items->sum('amount'),
                    'fees' => $items->sum('fee')
                ];
            });
            
        // تجميع البيانات حسب الحالة
        $byStatus = $remittances->groupBy('status')
            ->map(function ($items, $status) {
                return [
                    'status' => $status,
                    'count' => $items->count(),
                    'amount' => $items->sum('amount'),
                    'fees' => $items->sum('fee')
                ];
            });
            
        // تجميع البيانات حسب البلد
        $byCountry = $remittances->groupBy('destination_country')
            ->map(function ($items, $country) {
                return [
                    'country' => $country,
                    'count' => $items->count(),
                    'amount' => $items->sum('amount'),
                    'fees' => $items->sum('fee')
                ];
            });
            
        // الحصول على قائمة البلدان للفلتر
        $countries = Remittance::distinct('destination_country')->pluck('destination_country');
        
        return view('reports.remittances', compact(
            'remittances', 
            'startDate', 
            'endDate', 
            'status', 
            'destinationCountry',
            'totalCount',
            'totalAmount',
            'totalFees',
            'byCurrency',
            'byStatus',
            'byCountry',
            'countries'
        ));
    }

    /**
     * تقرير أسعار الصرف
     */
    public function exchangeRatesReport(Request $request)
    {
        // التحقق من المعلمات
        $startDate = $request->input('start_date') ? Carbon::parse($request->input('start_date')) : Carbon::now()->subMonths(3);
        $endDate = $request->input('end_date') ? Carbon::parse($request->input('end_date')) : Carbon::now();
        $fromCurrencyId = $request->input('from_currency_id');
        $toCurrencyId = $request->input('to_currency_id');
        
        // بناء الاستعلام
        $query = ExchangeRate::with(['fromCurrency', 'toCurrency'])
            ->whereBetween('effective_date', [$startDate->startOfDay(), $endDate->endOfDay()]);
            
        if ($fromCurrencyId) {
            $query->where('from_currency_id', $fromCurrencyId);
        }
        
        if ($toCurrencyId) {
            $query->where('to_currency_id', $toCurrencyId);
        }
        
        // الحصول على البيانات
        $exchangeRates = $query->orderBy('effective_date', 'desc')->get();
        
        // تجميع البيانات للرسم البياني
        $chartData = [];
        
        if ($fromCurrencyId && $toCurrencyId) {
            // تجميع البيانات لزوج عملات محدد
            $ratesByDate = $exchangeRates->groupBy(function ($item) {
                return $item->effective_date->format('Y-m-d');
            });
            
            foreach ($ratesByDate as $date => $rates) {
                $latestRate = $rates->sortByDesc('effective_date')->first();
                $chartData[] = [
                    'date' => $date,
                    'buy_rate' => $latestRate->buy_rate,
                    'sell_rate' => $latestRate->sell_rate,
                    'spread' => $latestRate->sell_rate - $latestRate->buy_rate
                ];
            }
        }
        
        // الحصول على قائمة العملات للفلتر
        $currencies = Currency::active()->ordered()->get();
        
        return view('reports.exchange_rates', compact(
            'exchangeRates', 
            'startDate', 
            'endDate', 
            'fromCurrencyId', 
            'toCurrencyId',
            'chartData',
            'currencies'
        ));
    }

    /**
     * تقرير الإيرادات
     */
    public function revenueReport(Request $request)
    {
        // التحقق من المعلمات
        $startDate = $request->input('start_date') ? Carbon::parse($request->input('start_date')) : Carbon::now()->startOfMonth();
        $endDate = $request->input('end_date') ? Carbon::parse($request->input('end_date')) : Carbon::now();
        $groupBy = $request->input('group_by', 'day');
        
        // تحديد تنسيق التاريخ وطريقة التجميع حسب المعلمة
        $dateFormat = '%Y-%m-%d';
        $dateInterval = '1 day';
        
        if ($groupBy == 'month') {
            $dateFormat = '%Y-%m';
            $dateInterval = '1 month';
        } elseif ($groupBy == 'year') {
            $dateFormat = '%Y';
            $dateInterval = '1 year';
        }
        
        // الحصول على إيرادات الحوالات
        $remittanceRevenue = Remittance::select(
                DB::raw("DATE_FORMAT(created_at, '{$dateFormat}') as date"),
                DB::raw('SUM(fee) as total_fees')
            )
            ->whereBetween('created_at', [$startDate->startOfDay(), $endDate->endOfDay()])
            ->groupBy('date')
            ->orderBy('date')
            ->get();
            
        // الحصول على إيرادات معاملات الصرف (يمكن تعديلها حسب هيكل جدول المعاملات)
        $exchangeRevenue = DB::table('transactions')
            ->select(
                DB::raw("DATE_FORMAT(created_at, '{$dateFormat}') as date"),
                DB::raw('SUM(fee) as total_fees'),
                DB::raw('SUM(profit) as total_profit')
            )
            ->whereBetween('created_at', [$startDate->startOfDay(), $endDate->endOfDay()])
            ->groupBy('date')
            ->orderBy('date')
            ->get();
            
        // دمج البيانات للرسم البياني
        $chartData = [];
        $dates = [];
        
        // إنشاء مجموعة من التواريخ المتسلسلة
        $currentDate = clone $startDate;
        while ($currentDate <= $endDate) {
            $dateKey = $currentDate->format($groupBy == 'day' ? 'Y-m-d' : ($groupBy == 'month' ? 'Y-m' : 'Y'));
            $dates[$dateKey] = [
                'remittance_fees' => 0,
                'exchange_fees' => 0,
                'exchange_profit' => 0
            ];
            
            if ($groupBy == 'day') {
                $currentDate->addDay();
            } elseif ($groupBy == 'month') {
                $currentDate->addMonth();
            } else {
                $currentDate->addYear();
            }
        }
        
        // إضافة بيانات الحوالات
        foreach ($remittanceRevenue as $item) {
            if (isset($dates[$item->date])) {
                $dates[$item->date]['remittance_fees'] = $item->total_fees;
            }
        }
        
        // إضافة بيانات الصرف
        foreach ($exchangeRevenue as $item) {
            if (isset($dates[$item->date])) {
                $dates[$item->date]['exchange_fees'] = $item->total_fees;
                $dates[$item->date]['exchange_profit'] = $item->total_profit;
            }
        }
        
        // تحويل البيانات للرسم البياني
        foreach ($dates as $date => $values) {
            $chartData[] = [
                'date' => $date,
                'remittance_fees' => $values['remittance_fees'],
                'exchange_fees' => $values['exchange_fees'],
                'exchange_profit' => $values['exchange_profit'],
                'total_revenue' => $values['remittance_fees'] + $values['exchange_fees'] + $values['exchange_profit']
            ];
        }
        
        // حساب الإحصائيات الإجمالية
        $totalRemittanceFees = array_sum(array_column($chartData, 'remittance_fees'));
        $totalExchangeFees = array_sum(array_column($chartData, 'exchange_fees'));
        $totalExchangeProfit = array_sum(array_column($chartData, 'exchange_profit'));
        $totalRevenue = $totalRemittanceFees + $totalExchangeFees + $totalExchangeProfit;
        
        return view('reports.revenue', compact(
            'chartData', 
            'startDate', 
            'endDate', 
            'groupBy',
            'totalRemittanceFees',
            'totalExchangeFees',
            'totalExchangeProfit',
            'totalRevenue'
        ));
    }

    /**
     * تقرير العملاء
     */
    public function customersReport(Request $request)
    {
        // التحقق من المعلمات
        $startDate = $request->input('start_date') ? Carbon::parse($request->input('start_date')) : Carbon::now()->startOfYear();
        $endDate = $request->input('end_date') ? Carbon::parse($request->input('end_date')) : Carbon::now();
        
        // الحصول على العملاء النشطين
        $customers = Customer::withCount([
                'remittances' => function ($query) use ($startDate, $endDate) {
                    $query->whereBetween('created_at', [$startDate->startOfDay(), $endDate->endOfDay()]);
                }
            ])
            ->orderBy('remittances_count', 'desc')
            ->take(20)
            ->get();
            
        // حساب إحصائيات إضافية للعملاء
        foreach ($customers as $customer) {
            // حساب إجمالي قيمة الحوالات
            $remittanceStats = Remittance::where('sender_id', $customer->id)
                ->whereBetween('created_at', [$startDate->startOfDay(), $endDate->endOfDay()])
                ->select(
                    DB::raw('SUM(amount) as total_amount'),
                    DB::raw('SUM(fee) as total_fees'),
                    DB::raw('COUNT(DISTINCT destination_country) as countries_count')
                )
                ->first();
                
            $customer->total_amount = $remittanceStats->total_amount;
            $customer->total_fees = $remittanceStats->total_fees;
            $customer->countries_count = $remittanceStats->countries_count;
            
            // الحصول على البلدان الأكثر إرسالاً إليها
            $topCountries = Remittance::where('sender_id', $customer->id)
                ->whereBetween('created_at', [$startDate->startOfDay(), $endDate->endOfDay()])
                ->select('destination_country', DB::raw('COUNT(*) as count'))
                ->groupBy('destination_country')
                ->orderBy('count', 'desc')
                ->take(3)
                ->pluck('destination_country')
                ->implode(', ');
                
            $customer->top_countries = $topCountries;
        }
        
        // إحصائيات العملاء العامة
        $totalCustomers = Customer::count();
        $newCustomers = Customer::whereBetween('created_at', [$startDate->startOfDay(), $endDate->endOfDay()])->count();
        $activeCustomers = Customer::whereHas('remittances', function ($query) use ($startDate, $endDate) {
            $query->whereBetween('created_at', [$startDate->startOfDay(), $endDate->endOfDay()]);
        })->count();
        
        return view('reports.customers', compact(
            'customers', 
            'startDate', 
            'endDate',
            'totalCustomers',
            'newCustomers',
            'activeCustomers'
        ));
    }

    /**
     * تقرير ملخص الأداء
     */
    public function performanceSummary()
    {
        // إحصائيات اليوم
        $today = Carbon::now()->startOfDay();
        $todayEnd = Carbon::now();
        
        $todayRemittances = Remittance::whereBetween('created_at', [$today, $todayEnd])->count();
        $todayCompletedRemittances = Remittance::whereBetween('created_at', [$today, $todayEnd])
            ->where('status', Remittance::STATUS_COMPLETED)
            ->count();
        $todayRevenue = Remittance::whereBetween('created_at', [$today, $todayEnd])->sum('fee');
        
        // إحصائيات الأسبوع
        $weekStart = Carbon::now()->startOfWeek();
        
        $weekRemittances = Remittance::whereBetween('created_at', [$weekStart, $todayEnd])->count();
        $weekCompletedRemittances = Remittance::whereBetween('created_at', [$weekStart, $todayEnd])
            ->where('status', Remittance::STATUS_COMPLETED)
            ->count();
        $weekRevenue = Remittance::whereBetween('created_at', [$weekStart, $todayEnd])->sum('fee');
        
        // إحصائيات الشهر
        $monthStart = Carbon::now()->startOfMonth();
        
        $monthRemittances = Remittance::whereBetween('created_at', [$monthStart, $todayEnd])->count();
        $monthCompletedRemittances = Remittance::whereBetween('created_at', [$monthStart, $todayEnd])
            ->where('status', Remittance::STATUS_COMPLETED)
            ->count();
        $monthRevenue = Remittance::whereBetween('created_at', [$monthStart, $todayEnd])->sum('fee');
        
        // إحصائيات أسعار الصرف
        $latestRates = ExchangeRate::with(['fromCurrency', 'toCurrency'])
            ->active()
            ->orderBy('updated_at', 'desc')
            ->take(10)
            ->get();
            
        // إحصائيات العملاء
        $topCustomers = Customer::withCount(['remittances'])
            ->orderBy('remittances_count', 'desc')
            ->take(5)
            ->get();
            
        // إحصائيات البلدان
        $topCountries = Remittance::select('destination_country', DB::raw('COUNT(*) as count'))
            ->groupBy('destination_country')
            ->orderBy('count', 'desc')
            ->take(5)
            ->get();
            
        return view('reports.performance', compact(
            'todayRemittances',
            'todayCompletedRemittances',
            'todayRevenue',
            'weekRemittances',
            'weekCompletedRemittances',
            'weekRevenue',
            'monthRemittances',
            'monthCompletedRemittances',
            'monthRevenue',
            'latestRates',
            'topCustomers',
            'topCountries'
        ));
    }

    /**
     * تصدير تقرير الحوالات إلى ملف CSV
     */
    public function exportRemittancesReport(Request $request)
    {
        // التحقق من المعلمات
        $startDate = $request->input('start_date') ? Carbon::parse($request->input('start_date')) : Carbon::now()->startOfMonth();
        $endDate = $request->input('end_date') ? Carbon::parse($request->input('end_date')) : Carbon::now();
        $status = $request->input('status');
        
        // بناء الاستعلام
        $query = Remittance::with(['sender', 'fromCurrency', 'toCurrency'])
            ->whereBetween('created_at', [$startDate->startOfDay(), $endDate->endOfDay()]);
            
        if ($status) {
            $query->where('status', $status);
        }
        
        // الحصول على البيانات
        $remittances = $query->orderBy('created_at', 'desc')->get();
        
        // إنشاء محتوى ملف CSV
        $csvContent = "رقم المرجع,تاريخ الإنشاء,المرسل,المستلم,بلد الوجهة,العملة المصدر,المبلغ,العملة الهدف,سعر الصرف,الرسوم,الحالة\n";
        
        foreach ($remittances as $remittance) {
            $csvContent .= "{$remittance->reference_number},{$remittance->created_at},{$remittance->sender->name},{$remittance->receiver_name},{$remittance->destination_country},{$remittance->fromCurrency->code},{$remittance->amount},{$remittance->toCurrency->code},{$remittance->exchange_rate},{$remittance->fee},{$remittance->status_name}\n";
        }
        
        // تحديد اسم الملف
        $fileName = "remittances_report_{$startDate->format('Y-m-d')}_{$endDate->format('Y-m-d')}.csv";
        
        // إنشاء استجابة لتنزيل الملف
        $headers = [
            'Content-Type' => 'text/csv; charset=UTF-8',
            'Content-Disposition' => "attachment; filename={$fileName}",
        ];
        
        return response($csvContent, 200, $headers);
    }
}
