<?php

namespace App\Http\Controllers;

use App\Models\Currency;
use App\Models\ExchangeRate;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class ExchangeRateController extends Controller
{
    /**
     * عرض قائمة أسعار الصرف
     */
    public function index()
    {
        $exchangeRates = ExchangeRate::with(['fromCurrency', 'toCurrency'])
            ->active()
            ->latest()
            ->get();
            
        $currencies = Currency::active()->ordered()->get();
        
        return view('exchange_rates.index', compact('exchangeRates', 'currencies'));
    }

    /**
     * عرض نموذج إضافة سعر صرف جديد
     */
    public function create()
    {
        $currencies = Currency::active()->ordered()->get();
        return view('exchange_rates.create', compact('currencies'));
    }

    /**
     * حفظ سعر صرف جديد في قاعدة البيانات
     */
    public function store(Request $request)
    {
        // التحقق من صحة البيانات المدخلة
        $validator = Validator::make($request->all(), [
            'from_currency_id' => 'required|exists:currencies,id',
            'to_currency_id' => 'required|exists:currencies,id|different:from_currency_id',
            'buy_rate' => 'required|numeric|min:0',
            'sell_rate' => 'required|numeric|min:0',
            'is_active' => 'boolean',
            'effective_date' => 'nullable|date',
            'notes' => 'nullable|string|max:500'
        ]);

        if ($validator->fails()) {
            return redirect()->route('exchange_rates.create')
                ->withErrors($validator)
                ->withInput();
        }

        // التحقق من عدم وجود سعر صرف مطابق
        $existingRate = ExchangeRate::where('from_currency_id', $request->from_currency_id)
            ->where('to_currency_id', $request->to_currency_id)
            ->where('effective_date', $request->effective_date ?? now())
            ->first();
            
        if ($existingRate) {
            return redirect()->route('exchange_rates.create')
                ->with('error', 'يوجد سعر صرف مطابق بالفعل لهذه العملات والتاريخ')
                ->withInput();
        }

        // إنشاء سجل سعر الصرف الجديد
        $exchangeRate = ExchangeRate::create([
            'from_currency_id' => $request->from_currency_id,
            'to_currency_id' => $request->to_currency_id,
            'buy_rate' => $request->buy_rate,
            'sell_rate' => $request->sell_rate,
            'is_active' => $request->has('is_active'),
            'effective_date' => $request->effective_date ?? now(),
            'updated_by' => Auth::id(),
            'notes' => $request->notes
        ]);

        return redirect()->route('exchange_rates.index')
            ->with('success', 'تم إضافة سعر الصرف بنجاح');
    }

    /**
     * عرض تفاصيل سعر صرف محدد
     */
    public function show(ExchangeRate $exchangeRate)
    {
        $exchangeRate->load(['fromCurrency', 'toCurrency', 'updatedBy']);
        
        // الحصول على سجل أسعار الصرف لنفس العملات
        $rateHistory = ExchangeRate::where('from_currency_id', $exchangeRate->from_currency_id)
            ->where('to_currency_id', $exchangeRate->to_currency_id)
            ->orderBy('effective_date', 'desc')
            ->get();
            
        return view('exchange_rates.show', compact('exchangeRate', 'rateHistory'));
    }

    /**
     * عرض نموذج تعديل سعر صرف محدد
     */
    public function edit(ExchangeRate $exchangeRate)
    {
        $currencies = Currency::active()->ordered()->get();
        return view('exchange_rates.edit', compact('exchangeRate', 'currencies'));
    }

    /**
     * تحديث بيانات سعر صرف محدد في قاعدة البيانات
     */
    public function update(Request $request, ExchangeRate $exchangeRate)
    {
        // التحقق من صحة البيانات المدخلة
        $validator = Validator::make($request->all(), [
            'from_currency_id' => 'required|exists:currencies,id',
            'to_currency_id' => 'required|exists:currencies,id|different:from_currency_id',
            'buy_rate' => 'required|numeric|min:0',
            'sell_rate' => 'required|numeric|min:0',
            'is_active' => 'boolean',
            'effective_date' => 'nullable|date',
            'notes' => 'nullable|string|max:500'
        ]);

        if ($validator->fails()) {
            return redirect()->route('exchange_rates.edit', $exchangeRate->id)
                ->withErrors($validator)
                ->withInput();
        }

        // التحقق من عدم وجود سعر صرف مطابق
        $existingRate = ExchangeRate::where('from_currency_id', $request->from_currency_id)
            ->where('to_currency_id', $request->to_currency_id)
            ->where('effective_date', $request->effective_date ?? now())
            ->where('id', '!=', $exchangeRate->id)
            ->first();
            
        if ($existingRate) {
            return redirect()->route('exchange_rates.edit', $exchangeRate->id)
                ->with('error', 'يوجد سعر صرف مطابق بالفعل لهذه العملات والتاريخ')
                ->withInput();
        }

        // تحديث بيانات سعر الصرف
        $exchangeRate->from_currency_id = $request->from_currency_id;
        $exchangeRate->to_currency_id = $request->to_currency_id;
        $exchangeRate->buy_rate = $request->buy_rate;
        $exchangeRate->sell_rate = $request->sell_rate;
        $exchangeRate->is_active = $request->has('is_active');
        $exchangeRate->effective_date = $request->effective_date ?? now();
        $exchangeRate->updated_by = Auth::id();
        $exchangeRate->notes = $request->notes;
        $exchangeRate->save();

        return redirect()->route('exchange_rates.index')
            ->with('success', 'تم تحديث سعر الصرف بنجاح');
    }

    /**
     * حذف سعر صرف محدد من قاعدة البيانات
     */
    public function destroy(ExchangeRate $exchangeRate)
    {
        // التحقق من عدم وجود معاملات مرتبطة بسعر الصرف
        // (يمكن إضافة هذا التحقق عند تطوير نموذج المعاملات)

        // حذف سعر الصرف
        $exchangeRate->delete();

        return redirect()->route('exchange_rates.index')
            ->with('success', 'تم حذف سعر الصرف بنجاح');
    }

    /**
     * تغيير حالة سعر الصرف (نشط/غير نشط)
     */
    public function toggleStatus(ExchangeRate $exchangeRate)
    {
        $exchangeRate->is_active = !$exchangeRate->is_active;
        $exchangeRate->updated_by = Auth::id();
        $exchangeRate->save();

        return redirect()->route('exchange_rates.index')
            ->with('success', 'تم تغيير حالة سعر الصرف بنجاح');
    }

    /**
     * الحصول على سعر الصرف بين عملتين محددتين
     */
    public function getRate(Request $request)
    {
        $fromCurrencyId = $request->input('from_currency_id');
        $toCurrencyId = $request->input('to_currency_id');
        
        if (!$fromCurrencyId || !$toCurrencyId) {
            return response()->json(['error' => 'يجب تحديد العملتين'], 400);
        }
        
        $exchangeRate = ExchangeRate::where('from_currency_id', $fromCurrencyId)
            ->where('to_currency_id', $toCurrencyId)
            ->active()
            ->latest()
            ->first();
            
        if (!$exchangeRate) {
            // محاولة البحث عن سعر الصرف العكسي
            $inverseRate = ExchangeRate::where('from_currency_id', $toCurrencyId)
                ->where('to_currency_id', $fromCurrencyId)
                ->active()
                ->latest()
                ->first();
                
            if ($inverseRate) {
                return response()->json([
                    'buy_rate' => $inverseRate->inverse_buy_rate,
                    'sell_rate' => $inverseRate->inverse_sell_rate,
                    'is_inverse' => true
                ]);
            }
            
            return response()->json(['error' => 'لا يوجد سعر صرف بين العملتين المحددتين'], 404);
        }
        
        return response()->json([
            'buy_rate' => $exchangeRate->buy_rate,
            'sell_rate' => $exchangeRate->sell_rate,
            'is_inverse' => false
        ]);
    }

    /**
     * تحديث أسعار الصرف بشكل جماعي
     */
    public function bulkUpdate(Request $request)
    {
        $rates = $request->input('rates', []);
        
        foreach ($rates as $rateData) {
            $exchangeRate = ExchangeRate::findOrFail($rateData['id']);
            $exchangeRate->buy_rate = $rateData['buy_rate'];
            $exchangeRate->sell_rate = $rateData['sell_rate'];
            $exchangeRate->updated_by = Auth::id();
            $exchangeRate->save();
        }
        
        return response()->json(['success' => true, 'message' => 'تم تحديث أسعار الصرف بنجاح']);
    }
}
