<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Services\AuthService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Redirect;

/**
 * وحدة التحكم بالمصادقة
 * تتعامل مع عمليات تسجيل الدخول والتسجيل وإدارة المستخدمين
 */
class AuthController extends Controller
{
    /**
     * خدمة المصادقة
     */
    protected $authService;
    
#
    /**
     * إنشاء نسخة جديدة من وحدة التحكم
     * 
     * @param AuthService $authService خدمة المصادقة
     */
    
    public function __construct(AuthService $authService)
    {
        $this->authService = $authService;

        // التحقق من تفعيل النظام - تم تعديله للسماح بالوصول إلى صفحة تسجيل الدخول مباشرة
        $this->middleware(function ($request, $next) {
            // السماح بالوصول إلى صفحة تسجيل الدخول حتى لو لم يكن النظام مفعلاً
            if ($request->is('login') || $request->is('register') || $request->is('activation') || $request->is('public/landing*')) {
                return $next($request);
            }
            
            return $next($request);
        });
        
        // التحقق من المصادقة
        $this->middleware('auth')->except(['showLoginForm', 'login', 'showRegistrationForm', 'register', 'showActivationForm', 'activate']);
    }
    
    /**
     * عرض نموذج تسجيل الدخول
     * 
     * @return \Illuminate\View\View
     */
    public function showLoginForm()
    {
        return view('auth.login');
    }
    
    /**
     * معالجة طلب تسجيل الدخول
     * 
     * @param Request $request طلب HTTP
     * @return \Illuminate\Http\RedirectResponse
     */
    public function login(Request $request){

        $validator = Validator::make($request->all(), [
            'email' => 'required|string',
            'password' => 'required|string',
        ]);
        
        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput($request->except('password'));
        }
        
        $remember = $request->has('remember');
        
        $result = $this->authService->login($request->email, $request->password, $remember);
        
        if (!$result['success']) {
            return redirect()->back()->with('error', $result['message'])->withInput($request->except('password'));
        }

        
        // توجيه المستخدم إلى لوحة التحكم بعد تسجيل الدخول بنجاح
        return redirect()->intended('/dashboard');
    }
    
    /**
     * تسجيل خروج المستخدم
     * 
     * @param Request $request طلب HTTP
     * @return \Illuminate\Http\RedirectResponse
     */
    public function logout(Request $request)
    {
        $this->authService->logout();
        
        // توجيه المستخدم إلى صفحة تسجيل الدخول بعد تسجيل الخروج
        return redirect('/login');
    }
    
    /**
     * عرض نموذج التسجيل
     * 
     * @return \Illuminate\View\View
     */
    public function showRegistrationForm()
    {
        return view('auth.register');
    }
    
    /**
     * معالجة طلب التسجيل
     * 
     * @param Request $request طلب HTTP
     * @return \Illuminate\Http\RedirectResponse
     */
    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'full_name' => 'required|string|max:255',
            'username' => 'required|string|max:50|unique:users',
            'email' => 'required|string|email|max:255|unique:users',
            'phone' => 'required|string|max:20',
            'address' => 'nullable|string|max:255',
            'password' => 'required|string|min:8|confirmed',
        ]);
        
        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput($request->except('password', 'password_confirmation'));
        }
        
        $userData = [
            'full_name' => $request->full_name,
            'username' => $request->username,
            'email' => $request->email,
            'phone' => $request->phone,
            'address' => $request->address,
            'password' => $request->password,
        ];
        
        $result = $this->authService->register($userData);
        
        if (!$result['success']) {
            return redirect()->back()
                ->with('error', $result['message'])
                ->withInput($request->except('password', 'password_confirmation'));
        }
        
        // توجيه المستخدم إلى صفحة تسجيل الدخول بعد التسجيل بنجاح
        return redirect('/login')->with('success', 'تم التسجيل بنجاح. يمكنك الآن تسجيل الدخول.');


    }
    


    /**
     * عرض نموذج تفعيل النظام
     * 
     * @return \Illuminate\View\View
     */
    public function showActivationForm(){
        return view('auth.activation');
    }



    
    /**
     * معالجة طلب تفعيل النظام
     * 
     * @param Request $request طلب HTTP
     * @return \Illuminate\Http\RedirectResponse
     */
    public function activate(Request $request){

        $validator = Validator::make($request->all(), [
            'activation_code' => 'required|string',
        ]);
        
        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }
        
        $result = $this->core->activate($request->activation_code);
        
        if (!$result) {
            return redirect()->back()->with('error', 'كود التفعيل غير صحيح. يرجى التحقق من الكود وإعادة المحاولة.')->withInput();
        }
        
        // توجيه المستخدم إلى صفحة تسجيل الدخول بعد تفعيل النظام بنجاح
        return redirect('/login')->with('success', 'تم تفعيل النظام بنجاح. يمكنك الآن تسجيل الدخول.');

    }
    
    /**
     * عرض لوحة التحكم
     * 
     * @return \Illuminate\View\View
     */
    public function dashboard()
    {
        $user = $this->authService->getCurrentUser();
        
        // يمكن إضافة المزيد من البيانات هنا لعرضها في لوحة التحكم
        
        return view('dashboard', compact('user'));
    }
}
